//UREBS-06 monitor program: MSP430FR2433
//sends the monitored information over TXD
// supports MAX burst length of 17 cycles
//monitors for:
//    button presses
//    phase measurements between spiral and loop resonances
//    ADC values of spiral and loop measured every cycle

/* IO configuration
-P2.2 burst in progress
-P3.0 used for troubleshooting output
-P3.1 not used yet
-mode:P2.7, UP:P2.4, DOWN:P2.3
-P1.0 LOOP edge, P1.1 LOOP ADC(A1)
-P1.4 SPIRAL egde, P1.5 SPIRAL ADC(A5)
-P1.6,P1.7 future ADC optional

typ 1% clock accuracy for 16MHz clock
Structure for data XFER : 71 bytes @ 112000 baud is about 6.3ms
0-mode (not used, must be input)
1-up  (not used, must be input)
2-down  (not used, must be input)
3-90 or 270 (contains count of # of occurrences of 270 found)
11-30 : phase measurements(20 max)
31-50: detector spiral (20 max)
51-70: detector loop (20 max)
 */

#include <msp430.h> 
#include <msp430fr2433.h>
#include <stdint.h>

//-----------[ Functions' Prototypes ]--------------
void delay_ms(unsigned int xdms);

#define TX_BYTES 71     //number of bytes to send to main MCU
uint8_t txData[TX_BYTES];
uint8_t i;              //general use
uint8_t flgBurst=0;     //flag to indicate burst in progress
uint8_t cntBurst=0, cntBurst2=0;     //count the number of burst (used as array index)
uint8_t flg270;         //flg to indicate 90(0) or 270(1)
uint8_t adc1,adc2;      //adc results
uint16_t cntMain=1;     //count main loops
uint16_t cntMainPrev=0;

// Expected value in ticks: 943 us / 4 us per tick = 235.75 -> round to 236 ticks
#define TICKS90DEG 377  //90 degs (then divid by 4 for approximation of 90 matching degrees)
#define TICKS180DEG 754 //180 degs
#define TICKS270DEG 1131
#define TICKS360DEG 1509 //1509*2.5us=3.77ms (one cycle of 265 Hz)

uint16_t tar, tar2;
volatile uint16_t measured_ticks = 0;
volatile int8_t  ph;    //phase difference measurement

volatile uint16_t start_time = 0;
volatile uint16_t stop_time = 0;
volatile int8_t time_error = 0;

static inline void sendBytes(const unsigned char *data, unsigned len) {
    for (i = 0; i < len; ++i) {
        while (!(UCA1IFG & UCTXIFG));
        UCA1TXBUF = data[i];
    }
}

#define wdt_start    (WDTPW + WDTTMSEL + WDTCNTCL + WDTSSEL__VLO + WDTIS__32K)  //  *** WATCHDOG TIMER MODE!!!!!  100us*WDTIS__32K= 3.2secs

int main(void) {
    WDTCTL = WDTPW | WDTHOLD;

    // operation beyond 8MHz requires wait state for memory
    FRCTL0 = FRCTLPW | NWAITS_1;

    __bis_SR_register(SCG0);                           // disable FLL
    CSCTL3 |= SELREF__REFOCLK;                         // Set REFO as FLL reference source
    CSCTL0 = 0;                                        // clear DCO and MOD registers
    CSCTL1 &= ~(DCORSEL_7);                            // Clear DCO frequency select bits first
    CSCTL1 |= DCORSEL_5;                               // Set DCO = 16MHz
    CSCTL2 = FLLD_0 + 487;                             // DCOCLKDIV = 16MHz
    __delay_cycles(3);
    __bic_SR_register(SCG0);                           // enable FLL
    while(CSCTL7 & (FLLUNLOCK0 | FLLUNLOCK1));         // FLL locked

    WDTCTL = wdt_start; // start the Watchdog

    P1DIR = 0x00;  //all inputs
    // --- Port 1 interrupts on rising edges ---
    P1IFG &= ~(BIT0 | BIT4);   // clear any pending flags
    P1IES &= ~(BIT0 | BIT4);   // 0 = low->high (rising edge)
    P1IE  |=  (BIT0 | BIT4);   // enable interrupts

    //UART SETUP: UCA1TXD=P2.5, UCA1RXD=P2.6
    P2SEL0 |= BIT5 | BIT6;
    P2SEL1 &= ~(BIT5 | BIT6);
    P2DIR = 0x00;  //all inputs,  P2.7, P2.4, P2.3 as inputs
    //configure P2.2 for burst detection from main MCU
    P2IE  |= BIT2;      // enable interrupt on P2.2
    P2IES &= ~BIT2;     // start with rising-edge detection
    P2IFG &= ~BIT2;     // clear any pending flag

    //Port 3
    P3DIR = 0x01;  //Bit 0 output (all others inputs) (P3.1 input)

    // Enable analog function on P1.1 (A1) and P1.5 (A5) and 6,7 for future use
    SYSCFG2 |= ADCPCTL1 | ADCPCTL5 | ADCPCTL6 | ADCPCTL7;

    // Configure ADC10
    ADCCTL0 |= ADCSHT_8 | ADCON;           // ADCON, S&H=256 ADC clks (modified inline in program)
    ADCCTL1 |=  ADCSSEL_3 | ADCDIV_4 | ADCSHP;  //ADCCLK = SMCLK16MHz/4 +ADC sample-and-hold pulse-mode select
    //ADCCTL2 |= ADCRES;                     // 10-bit conversion results
    ADCCTL2 = ADCRES_0;                      // 8-bit resolution
   // ADCIE |= ADCIE0;                       // Enable ADC conv complete interrupt
    ADCMCTL0 = ADCINCH_1 | ADCSREF_1;        // default to P1.1, Vref=1.5V
    PMMCTL0_H = PMMPW_H;                     // Unlock the PMM registers
    PMMCTL2 |= INTREFEN;                    // Enable internal reference (ensures enabled)

    // --- Timer_A0 setup: SMCLK = 16MHz, continuous, divider /8/5 => 2.5 us ticks --- 1509/cycle of 265 Hz
    TA0CTL = TASSEL__SMCLK | MC__CONTINUOUS | TACLR | ID__8; // SMCLK, cont, clear, /8 main
    TA0EX0 = TAIDEX_4;  //divid by 5

    // Configure
    PMMCTL0_H = PMMPW_H;                  // Unlock the PMM registers
    PM5CTL0 &= ~LOCKLPM5;                // Disable the GPIO power-on default high-impedance mode
                                     // to activate previously configured port settings

    // UART SETUP for ~112000 baud, SMCLK = 16 MHz
    UCA1CTLW0 = UCSWRST;                   // Reset eUSCI_A1
    UCA1CTLW0 |= UCSSEL__SMCLK;            // SMCLK = 16MHz
    UCA1BRW   = 8;                         // Integer divider
    UCA1MCTLW = UCOS16 | (15 << 4) | (0 << 8);  // UCOS16=1, UCBRF=15, UCBRS=0
    UCA1CTLW0 &= ~UCSWRST;                 // Enable eUSCI_A1

 //   __bis_SR_register(GIE);     // Enable global interrupts
    __enable_interrupt();
    SFRIE1 |= WDTIE;   // enable Watchdog Timer interrupt

    while (1) {
        //interrupt driven program
        cntMain++; //increment loop counter to ensure program is not hung (checked in watchdog)
        __delay_cycles(24*1000);   //1ms delay (1000 counts/sec)
    }
}

//********************************************************************


//******************************
// Port 1 ISR: handle P1.0 rising (clear timer) and P1.4 rising (read timer)
#pragma vector = PORT1_VECTOR
__interrupt void Port1_ISR(void)
{

    int16_t err_ticks = 0;

    // Start event (rising on P1.0-loop): clear timer
    if (P1IFG & BIT0) {
        // Clear TAR (timer counter). TACLR is write-one-to-clear.
        tar2 = TA0R;            //save before clearing
        TA0CTL |= TACLR;        // clear timer immediately
        P3OUT |= BIT0;  //set high
        // --- Read channel A1 (loop)
        ADCCTL0 &= ~ADCENC;               //disable ADC conversion, to allow changing channel in register
        ADCCTL0 = ADCSHT_8 | ADCON;       // ADCON, S&H=256 ADC clks  4MHz*256= 64us per sample?
        ADCMCTL0 = ADCINCH_1 | ADCSREF_1; // A1(P1.1) (ref 1.5V)
        ADCCTL0 |= ADCENC | ADCSC;          // Sampling and conversion start
        while (ADCCTL1 & ADCBUSY);
        adc1=ADCMEM0;          // 8-bit result
        txData[(50+cntBurst2)] = adc1;          // 8-bit result

        if (P2IN&BIT2){
            //in burst
            cntBurst2++;  //increment seperate burst counter so also works for vert only
            if (cntBurst2>20) cntBurst2=20;  //limit
        } else {
            cntBurst2=0;  //reset
        }

        P1IFG &= ~BIT0;          // clear flag
    }

    // Stop event (rising on P1.4-spiral): read timer value
    if (P1IFG & BIT4) {
        // Read TAR (TA0R)  cycles since last TACLR
        tar = TA0R;
        P3OUT &= ~BIT0;  //allows scoping to confirm timing
        measured_ticks = tar;
       // if (measured_ticks<TICKS360DEG){
            //only valid if measured less than one cycle
            if (measured_ticks>TICKS180DEG){
                err_ticks = (int16_t)measured_ticks - (int16_t)TICKS270DEG;
                flg270=1;
            } else {
                // difference is less than 180 deg
                // compute error in ticks relative to EXPECTED_TICKS
                err_ticks = (int16_t)measured_ticks - (int16_t)TICKS90DEG;
                flg270=0;
            }
            // saturate to int8 range
            err_ticks=err_ticks>>2;   //divid by 4 for degree approximation
            if (err_ticks > 127) err_ticks = 127;
            if (err_ticks < -128) err_ticks = -128;
            if (flgBurst){
                cntBurst++;     //increment Burst count
                if (cntBurst>=20) cntBurst=20;  //limit to avoid array overflow
                ph = (int8_t)(err_ticks);   // signed approx degree error
                //if flg270 was set then increment txData[3]
                 if (flg270) txData[3]++;
                 //save the phase measurement
                 txData[(10+cntBurst)]=ph;
            }

            //perform ADC readings on (A5-spiral)
            // --- Read channel A5 (spiral)
            ADCCTL0 &= ~ADCENC;               //disable ADC conversion, to allow changing channel in register
            ADCCTL0 = ADCSHT_8 | ADCON;       // ADCON, S&H=256 ADC clks  4MHz*256= 64us per sample?
            ADCMCTL0 = ADCINCH_5 | ADCSREF_1; // A5(P1.5) (ref 1.5V)
            ADCCTL0 |= ADCENC | ADCSC;          // Sampling and conversion start
            while (ADCCTL1 & ADCBUSY);
            adc2=ADCMEM0;          // 8-bit result
            txData[(30+cntBurst)] = adc2;          // 8-bit result


      //  }
        P1IFG &= ~BIT4;          // clear flag
    }
}

//******************************
// Port 2 ISR: handle P2.2
#pragma vector=PORT2_VECTOR
__interrupt void Port_2_ISR(void)
{
    if (P2IFG & BIT2) {
        // Check current edge sense
        if (P2IES & BIT2) {
            // Was set for falling - falling edge occurred
            flgBurst=0;         //burst done
            cntBurst2=0;
            //need to send measured data to main MCU
            sendBytes(txData, TX_BYTES);  //for 60 bytes takes about 11ms
            __no_operation();     //
            //clear the txData array so ready to be refilled
            for (i = 3; i < TX_BYTES; ++i) {
                txData[i]=0;  //clear the array element
            }
            //P3OUT &= ~BIT0;  //set low
            P2IES &= ~BIT2;   // switch to rising edge for next time
        } else {
            // Was set for rising - rising edge occurred
            //if the pulse is less than 1us wide then send the button press information
            __delay_cycles(16);
            if (P2IN&BIT2) {
                //still high so then signal is for start of burst
                flgBurst=1;     //mark as burst detected
                cntBurst=0;  //reset burst counter
                P2IES |= BIT2;    // switch to falling edge for next time
            } else {
                //P2.2 dropped low mean it is signal to send button presses
                sendBytes(txData, 3);  //for 3 bytes takes < 1ms
                //clear the txData array so ready to be refilled
                for (i = 0; i < 3; ++i) {
                    txData[i]=0;  //clear only the button press info
                }
                P2IFG &= ~BIT7; P2IFG &= ~BIT4; P2IFG &= ~BIT3;  // clear interrupt flags
                //leave the interrupt on rising edge
            }

            //P3OUT |= BIT0;  //set high

        }
        P2IFG &= ~BIT2;       // clear interrupt flag
    }
}


// *******************   WDT INTERRUPT **********************
// Watchdog Timer interrupt service routine- Timer Mode
#pragma vector=WDT_VECTOR
__interrupt void WDT_ISR(void)
{
    //verify program working by checking cntMain is incrementing
    // typically counter will increment 1000/sec so in 3.2secs = 3200 advances (<16bit value)
    if (cntMain==cntMainPrev) {
        //MCU hung, so MCU needs reboot
        PMMCTL0 |= PMMSWBOR;    //cause a software brown out reset
    } else {
        cntMainPrev=cntMain;
    }
}//WD interrupt




//---------------------------------------------------------------------

//****************** delay_ms ******************
void delay_ms(unsigned int xdms)
{
    unsigned int xi;
    for (xi = xdms; xi > 0; xi--) {
        __delay_cycles(16*1000);   //1ms delay
    }
}

